
rm(list=ls())

source("Packages.R")

###################################################################################################################
####################### READ AND PREPARE VARS #############################################################
###################################################################################################################

dat = read.dta13("data_december2020.dta",
                 convert.factors = FALSE)

dat = zap_label(dat)

names(dat) = tolower(names(dat))

dat$age = dat$v_1
dat$female = as.numeric(dat$gender == 2)

dat$edu = car::recode(dat$v_22, "1=1; 2=1; 3=1; 4=2; 5=3; 6=3; 7=3; 8=3; 9=NA")
dat$income = dat$pendapatan

dat$urban = as.numeric(dat$desakota == 2)

dat$ethnicity = dat$etnis_rec

dat$muslim = as.numeric(dat$agama_rec == 1)

dat$group = gsub(" ", "", dat$exp_1_21b, fixed = TRUE)
dat$group = car::recode(dat$group, "'KELOMPOKA' = 'Control';
                        'KELOMPOKB' = 'Natuna';
                        'KELOMPOKC' = 'Uighurs';
                        'KELOMPOKD' = 'Investment'")
dat$group = factor(dat$group, levels=c("Control", "Natuna", "Uighurs", "Investment"))

dat$treat.hoax = car::recode(dat$v_21b,"1=0; 2=1; else=NA")



## dependent variables
dat$loyalty = car::recode(dat$v_21c_a, "8=NA")
dat$selfinterested = car::recode(dat$v_21c_b, "8=NA")
dat$culture = car::recode(dat$v_21c_c, "8=NA")
dat$greedy = car::recode(dat$v_21c_d, "8=NA")
dat$influence = car::recode(dat$v_21c_e, "8=NA")

dat$negtraits = rowMeans(dat[,c("loyalty", "selfinterested", "culture",
                         "greedy", "influence")], na.rm=TRUE)


## beneficial
dat$beneficial = car::recode(dat$v_21d, "8=NA")


## positive/negative
dat$posneg = car::recode(dat$v_21e, "2=3; 3=2; 8=NA")


# Feeling: ethnic Chinese
dat$feelchinese = car::recode(dat$v_10a_a, "8=NA")
# Feeling: Christian
dat$feelchristian = car::recode(dat$v_10a_b, "8=NA")
# Feeling: Muslim
dat$feelmuslim = car::recode(dat$v_10a_c, "8=NA")

## 3-level feeling chinese
dat$feelchinese_3lvl = car::recode(dat$feelchinese, "1='Negative'; 2='Negative'; 3='Neutral'; 4='Positive'; 5='Positive'")
dat$feelchinese_3lvl = factor(dat$feelchinese_3lvl, levels = c("Neutral", "Negative", "Positive"))


###################################################################################################################
####################### PREPARE SURVEY AND RUN PCA ##########################################################
###################################################################################################################

### Create Survey Data
svydat = svydesign(id=~1, data=dat, weights = dat$bobot_pakai)
svydat.muslim = subset(svydat, subset=(muslim==1))
dat.muslim = subset(dat, subset=(muslim==1))

### PCA of dependent variables
pca = svyprcomp(~loyalty + selfinterested + culture + greedy + influence, svydat.muslim)
summary(pca)
pca
psych::alpha(dat.muslim[, c("loyalty", "selfinterested", "culture", "greedy", "influence")])


###################################################################################################################
### Main REGRESSIONS 
###################################################################################################################

### Main Effects
svylm.negtraits = svyglm(negtraits ~ group, design = svydat.muslim, family="gaussian")
summary(svylm.negtraits)

svylm.beneficial = svyglm(beneficial ~ group, design = svydat.muslim, family="gaussian")
summary(svylm.beneficial)

svylm.posneg = svyglm(posneg ~ group, design = svydat.muslim, family="gaussian")
summary(svylm.posneg)

svylm.hoax = svyglm(treat.hoax ~ group, design = svydat.muslim, family="gaussian")
summary(svylm.hoax)


### Interaction with Positive/Negative
svylm.negtraits.posneg = svyglm(negtraits ~ group*posneg, design = svydat.muslim, family="gaussian")
summary(svylm.negtraits.posneg)

svylm.beneficial.posneg = svyglm(beneficial ~ group*posneg, design = svydat.muslim, family="gaussian")
summary(svylm.beneficial.posneg)


###################################################################################################################
## PLOT REGRESSION COEFFICIENTS
###################################################################################################################

plot.beneficial = 
  plot_summs(svylm.beneficial,
             coefs=c("Territorial\nViolation"="groupNatuna",
                     "Uighurs\nDiscrimination" = "groupUighurs",
                     "Foreign\nInvestment"="groupInvestment"),
             colors = "black",
             inner_ci_level = .90) +
  ggtitle("Perceived Benefits of\na Close Tie with China") +
  theme(plot.title=element_text(size=10, face="bold", hjust = 0.5),
        panel.border = element_rect(fill=NA, size=0.5)) +
  geom_vline(xintercept = 0, color="red", size=0.4) +
  coord_cartesian(xlim=c(-.4, .4)) 



plot.negtraits = 
  plot_summs(svylm.negtraits,
             coefs=c("Territorial\nViolation"="groupNatuna",
                     "Uighurs\nDiscrimination" = "groupUighurs",
                     "Foreign\nInvestment"="groupInvestment"),
             colors = "black",
             inner_ci_level = .90) +
  ggtitle("Negative Stereotypes\ntoward Ethnic Chinese") +
  theme(plot.title=element_text(size=10, face="bold", hjust = 0.5),
        axis.text.y = element_blank(),
        panel.border = element_rect(fill=NA, size=0.5),
        plot.margin = unit(c(0,1,0,1),"cm"))  +
  geom_vline(xintercept = 0, color="red", size=0.4) +
  coord_cartesian(xlim=c(-.4, .4)) 


plot.posneg = 
  plot_summs(svylm.posneg,
             coefs=c("Territorial\nViolation"="groupNatuna",
                     "Uighurs\nDiscrimination" = "groupUighurs",
                     "Foreign\nInvestment"="groupInvestment"),
             colors = "black",
             inner_ci_level = .90) +
  ggtitle("Perceived Valence of the News\n(Negative to Positive)") +
  theme(plot.title=element_text(size=10, face="bold", hjust = 0.5),
        axis.text.y = element_blank(),
        panel.border = element_rect(fill=NA, size=0.5))  +
  geom_vline(xintercept = 0, color="red", size=0.4) +
  coord_cartesian(xlim=c(-.65, .65)) 


png("Figure 1b.png", width=3000, height=2000, res=300)
plot.beneficial + plot.negtraits + plot.posneg +
  plot_annotation(title = 'Marginal Effects of the Treatments (December 2020 Survey)')

dev.off()



###################################################################################################################
## ROBUSTNESS: INCLUDE NON-MUSLIMS
###################################################################################################################

### Include Non-Muslims
svylm.beneficial.nonmuslim = svyglm(beneficial ~ group, 
                                    design = svydat, family = "gaussian")
summary(svylm.beneficial.nonmuslim)

svylm.negtraits.nonmuslim = svyglm(negtraits ~ group, 
                                   design = svydat, family = "gaussian")
summary(svylm.negtraits.nonmuslim)

svylm.posneg.nonmuslim = svyglm(posneg ~ group, 
                                    design = svydat, family = "gaussian")
summary(svylm.posneg.nonmuslim)


###################################################################################################################
## ROBUSTNESS: ETHNIC CHINESE BARPLOT AND LINEAR
###################################################################################################################

### Barplot of Feeling of Ethnic Chinese
temp = data.frame(round(prop.table(table(dat.muslim$feelchinese))*100, 2))
colnames(temp) = c("Feeling", "Percentage")
temp$Feeling = c("Strong Dislike", "Dislike", "Neutral", "Like", "Strong Liking")
temp$Feeling = factor(temp$Feeling, 
                      c("Strong Dislike", "Dislike", "Neutral", "Like", "Strong Liking"))

plot_feelchinese = ggplot(data=temp, aes(x=Feeling, y=Percentage)) +
  geom_bar(stat="identity")+
  geom_text(aes(label=Percentage), vjust=-0.3, size=3.5)+
  theme_minimal() + 
  theme(axis.title.x = element_blank(),
        plot.title=element_text(size=10)) + 
  coord_cartesian(ylim=c(0, 80)) +
  ggtitle("Distribution of Feeling toward Ethnic Chinese, December 2020")

png("Figure 2b.png", width=2000, height=1000, res=300)
plot_feelchinese
dev.off()



###################################################################################################################
## ROBUSTNESS: ETHNIC CHINESE CATEGORICAL
###################################################################################################################

plot.int2 <- function(plot.main, title, obj, obj.label, pos.legend, legend.title.size, legend.label.size, lab.y, ylim, size.x=15, size.y=15) {
  plot.out <-
    plot.main +
    geom_errorbar(aes(ymin=Estimate - 1.96*se, ymax=Estimate + 1.96*se, col=obj), 
                  width=.1, lwd=1, position = position_dodge(width = .3)) +
    geom_errorbar(aes(ymin=Estimate - 1.64*se, ymax=Estimate + 1.64*se, col=obj), 
                  width=.1, lwd=2, position = position_dodge(width = .3)) +
    geom_point(aes(col=obj, pch=obj), 
               size=4, fill="black", position = position_dodge(width = .3)) +
    theme_bw() + 
    ylab(lab.y) +
    xlab("") +
    theme(axis.title.x = element_text(size = size.x)) +
    theme(axis.title.y = element_text(size = size.y)) +
    theme(axis.text.x = element_text(size = size.x)) +
    theme(axis.text.y = element_text(size = size.y)) +
    coord_cartesian(ylim=c(ylim[1], ylim[2])) + 
    theme(legend.text=element_text(size=legend.label.size)) + 
    guides(pch=guide_legend(obj.label, title.theme = element_text(size=legend.title.size), reverse=TRUE), 
           col=guide_legend(obj.label, title.theme = element_text(size=legend.title.size), reverse=TRUE)) +
    theme(legend.position = pos.legend) + 
    theme(legend.background = element_rect(fill="white", 
                                           size=.1, linetype="solid")) + 
    theme(plot.title = element_text(size = 10)) + 
    ggtitle(title)
  
  return (plot.out)
}

### Create Data
temp_beneficial = dat.muslim[complete.cases(dat.muslim[, c("group", "beneficial", "feelchinese_3lvl")]), ]
svytemp_beneficial = svydesign(id = ~1, data = temp_beneficial, weights = temp_beneficial$weight)

temp_negtraits = dat.muslim[complete.cases(dat.muslim[, c("group", "negtraits", "feelchinese_3lvl")]), ]
svytemp_negtraits = svydesign(id = ~1, data = temp_negtraits, weights = temp_negtraits$weight)

temp_posneg = dat.muslim[complete.cases(dat.muslim[, c("group", "beneficial", "feelchinese_3lvl")]), ]
svytemp_posneg = svydesign(id = ~1, data = temp_posneg, weights = temp_posneg$weight)


### Regression Benefits
int.beneficial.chinese.cat = svyglm(beneficial ~ group*feelchinese_3lvl, 
                                    design = svytemp_beneficial, family = "gaussian")
summary(int.beneficial.chinese.cat)

out_beneficial = summary(margins(int.beneficial.chinese.cat, design = svytemp_beneficial, at = list(feelchinese_3lvl = c("Neutral", "Negative", "Positive"))))
out_beneficial = out_beneficial[out_beneficial$factor %in% c("groupInvestment", "groupNatuna", "groupUighurs"), c("factor", "feelchinese_3lvl", "AME", "SE")]

colnames(out_beneficial) = c("Treatment", "Feeling", "Estimate", "se")
out_beneficial$Treatment = c("Foreign\nInvestment","Foreign\nInvestment","Foreign\nInvestment",
                             "Territorial\nViolation","Territorial\nViolation","Territorial\nViolation",
                             "Uighurs\nDiscrimination","Uighurs\nDiscrimination","Uighurs\nDiscrimination")
out_beneficial$Treatment = factor(out_beneficial$Treatment, levels = c("Foreign\nInvestment", "Uighurs\nDiscrimination", "Territorial\nViolation"))
out_beneficial$Feeling = factor(out_beneficial$Feeling, levels = c("Neutral", "Negative", "Positive"))

plot.temp = ggplot(out_beneficial, aes(Treatment, Estimate))
plot_beneficial = plot.int2(plot.temp, 
                            title="Perceived Benefits",
                            obj=out_beneficial$Feeling,
                            obj.label = "Feeling toward Ethnic Chinese",
                            pos.legend = c(.87, .1),
                            legend.title.size = 9,
                            legend.label.size = 9,
                            lab.y = "Estimate",
                            ylim = c(-.9, .2),
                            size.x = 10,
                            size.y = 10) +
  geom_hline(yintercept=0, linetype="dashed") +
  coord_flip()


### Regression Negative Traits
int.negtraits.chinese.cat = svyglm(negtraits ~ group*feelchinese_3lvl, 
                                   design = svytemp_negtraits, family = "gaussian")
summary(int.negtraits.chinese.cat)

out_negtraits = summary(margins(int.negtraits.chinese.cat, design = svytemp_negtraits, at = list(feelchinese_3lvl = c("Neutral", "Negative", "Positive"))))
out_negtraits = out_negtraits[out_negtraits$factor %in% c("groupInvestment", "groupNatuna", "groupUighurs"), c("factor", "feelchinese_3lvl", "AME", "SE")]

colnames(out_negtraits) = c("Treatment", "Feeling", "Estimate", "se")
out_negtraits$Treatment = c("Foreign\nInvestment","Foreign\nInvestment","Foreign\nInvestment",
                            "Territorial\nViolation","Territorial\nViolation","Territorial\nViolation",
                            "Uighurs\nDiscrimination","Uighurs\nDiscrimination","Uighurs\nDiscrimination")
out_negtraits$Treatment = factor(out_negtraits$Treatment, levels = c("Foreign\nInvestment", "Uighurs\nDiscrimination", "Territorial\nViolation"))
out_negtraits$Feeling = factor(out_negtraits$Feeling, levels = c("Neutral", "Negative", "Positive"))

plot.temp = ggplot(out_negtraits, aes(Treatment, Estimate))
plot_negtraits = plot.int2(plot.temp, 
                           title="Negative Stereotypes",
                           obj=out_negtraits$Feeling,
                           obj.label = "Feeling toward Ethnic Chinese",
                           pos.legend = c(.87, .1),
                           legend.title.size = 9,
                           legend.label.size = 9,
                           lab.y = "Estimate",
                           ylim = c(-.9, .2),
                           size.x = 10,
                           size.y = 10) +
  geom_hline(yintercept=0, linetype="dashed") +
  coord_flip()





### Regression Positive Negative
int.posneg.chinese.cat = svyglm(posneg ~ group*feelchinese_3lvl, 
                                    design = svytemp_posneg, family = "gaussian")
summary(int.posneg.chinese.cat)

out_posneg = summary(margins(int.posneg.chinese.cat, design = svytemp_posneg, at = list(feelchinese_3lvl = c("Neutral", "Negative", "Positive"))))
out_posneg = out_posneg[out_posneg$factor %in% c("groupInvestment", "groupNatuna", "groupUighurs"), c("factor", "feelchinese_3lvl", "AME", "SE")]

colnames(out_posneg) = c("Treatment", "Feeling", "Estimate", "se")
out_posneg$Treatment = c("Foreign\nInvestment","Foreign\nInvestment","Foreign\nInvestment",
                             "Territorial\nViolation","Territorial\nViolation","Territorial\nViolation",
                             "Uighurs\nDiscrimination","Uighurs\nDiscrimination","Uighurs\nDiscrimination")
out_posneg$Treatment = factor(out_posneg$Treatment, levels = c("Foreign\nInvestment", "Uighurs\nDiscrimination", "Territorial\nViolation"))
out_posneg$Feeling = factor(out_posneg$Feeling, levels = c("Neutral", "Negative", "Positive"))

plot.temp = ggplot(out_posneg, aes(Treatment, Estimate))
plot_posneg = plot.int2(plot.temp, 
                            title="Valence of the News",
                            obj=out_posneg$Feeling,
                            obj.label = "Feeling toward Ethnic Chinese",
                            pos.legend = c(.87, .1),
                            legend.title.size = 9,
                            legend.label.size = 9,
                            lab.y = "Estimate",
                            ylim = c(-.9, .2),
                            size.x = 10,
                            size.y = 10) +
  geom_hline(yintercept=0, linetype="dashed") +
  coord_flip()


png(filename = "Figure 3b.png", width=3000, height=2000, res=300)
(plot_beneficial + plot_negtraits + plot_posneg) +
  plot_annotation(
    title = 'Marginal Effects by Feeling toward Ethnic Chinese, December 2020') +
  plot_layout(guides = "collect") & theme(legend.position = "bottom")
dev.off()



###################################################################################################################
## OUTPUT REGRESSION MODELS
###################################################################################################################

tab_model(svylm.negtraits,
          svylm.beneficial,
          svylm.posneg,
          show.se = TRUE, collapse.se = TRUE,
          p.val = "wald",
          p.style = "stars",
          show.ci = FALSE, show.p = FALSE,
          file="Regression December 2020 (Main).htm")


tab_model(svylm.hoax,
          svylm.negtraits.posneg,
          svylm.beneficial.posneg,
          svylm.beneficial.nonmuslim,         
          svylm.negtraits.nonmuslim,
          svylm.posneg.nonmuslim,
          int.negtraits.chinese.cat,
          int.beneficial.chinese.cat,
          int.posneg.chinese.cat,
          show.se = TRUE, collapse.se = TRUE,
          p.val = "wald",
          p.style = "stars",
          show.ci = FALSE, show.p = FALSE,
          file="Regression December 2020 (Robustness).htm")


###################################################################################################################
## COVARIATES BALANCE
###################################################################################################################

sink("Balance Check December 2020.txt")
# female
print("Female ==========")
summary(aov(female ~ group, subset(dat, muslim==1)))

# age
print("Age ==========")
summary(aov(age ~ group, subset(dat, muslim==1)))

# edu
print("Education ==========")
summary(aov(edu ~ group, subset(dat, muslim==1)))

# income
print("Income ==========")
summary(aov(income ~ group, subset(dat, muslim==1)))

# urban/rural
print("Urban/Rural ==========")
summary(aov(urban ~ group, subset(dat, muslim==1)))

# feeling: chinese
print("Feeling Chinese ==========")
summary(aov(feelchinese ~ group, subset(dat, muslim==1)))

sink(NULL)
